<?php

namespace App\Console\Commands;

use App\Models\UserCourse;
use App\Models\VideoProgress;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class UpdateCourseProgress extends Command
{
    protected $signature = 'course:progress';
    protected $description = 'Update course progress for all users based on completed lessons';

    public function handle()
    {
        $enrollments = UserCourse::with('course.sections.lessons.videoProgress')->get();

        foreach ($enrollments as $enrollment) {
            $totalLessons = $enrollment->course->sections->flatMap->lessons->count();
            Log::info("Processing enrollment for User {$enrollment->User_id}, Course {$enrollment->Course_id}, Total Lessons: {$totalLessons}");

            if ($totalLessons === 0) {
                Log::warning("No lessons found for Course {$enrollment->Course_id}");
                continue;
            }

            $completedLessons = $enrollment->course->sections->flatMap->lessons
                ->map(function ($lesson) use ($enrollment) {
                    $progress = $lesson->videoProgress->where('User_id', $enrollment->User_id)->first();
                    $isCompleted = $progress && $progress->Is_completed ? 1 : 0;
                    Log::info("Lesson {$lesson->Lesson_id} for User {$enrollment->User_id}: " . ($isCompleted ? 'Completed' : 'Not Completed'), [
                        'Progress_id' => $progress ? $progress->Progress_id : null,
                        'Is_completed' => $progress ? $progress->Is_completed : null,
                    ]);
                    return $isCompleted;
                })->sum();

            $progressPercentage = $totalLessons > 0 ? ($completedLessons / $totalLessons) * 100 : 0;
            $status = $progressPercentage >= 100 ? 'completed' : ($progressPercentage > 0 ? 'in_progress' : 'enrolled');
            $completionStatus = $progressPercentage >= 100 ? 'completed' : ($progressPercentage > 0 ? 'in_progress' : 'enrolled');

            $enrollment->update([
                'Progress' => number_format($progressPercentage, 2),
                'Status' => $status,
                'Completion_status' => $completionStatus,
                'Last_activity' => now(),
                'Total_attempts' => $enrollment->Total_attempts + 1,
                'updated_at' => now(),
            ]);

            Log::info("Updated progress for User {$enrollment->User_id}, Course {$enrollment->Course_id}: {$progressPercentage}%");
        }

        $this->info('Course progress updated successfully.');
    }
}
